| package |
package := Package name: 'Sport'.
package paxVersion: 1;
	basicComment: 'Sport is a Smalltalk Portability library.
All Sport implementations are available under the LGPL.

Sport provides a consistent interface to things which are implemented differently in most Smalltalk dialects. Sport provides wrappers for:
    * Exceptions
    * Times & Dates
    * Files
    * Sockets
     ... and more.

So far, Sport is implemented in: 
 - VisualWorks
 - Gemstone
 - Squeak
 - Visual Age for Smalltalk
 - Dolphin Smalltalk X6

Website: http://wiki.cs.uiuc.edu/VisualWorks/Sport

===============
Dolphin X6 port by:
Esteban A. Maringolo <eMaringolo at gmail dot com>
Jose Sebastian Calvo <fxgallego at gmail dot com>
working at InfOil S.A.

This work was done based on the previous port of Esteban A. Maringolo
and the newest version loaded from the Seaside 2.9 alpha 4 Sport for Squeak
package on its version 2.031

It may include methods not yet implemented, and other unknown errors. 
If you fix some, please report them to my mail address. Thank you.

================
'.

package basicPackageVersion: 'Sport 0.004'.


package classNames
	add: #SpAbstractError;
	add: #SpDate;
	add: #SpEnvironment;
	add: #SpError;
	add: #SpExceptionContext;
	add: #SpFilename;
	add: #SpFileStream;
	add: #SpIPAddress;
	add: #SpSocket;
	add: #SpSocketAddress;
	add: #SpSocketError;
	add: #SpStringUtilities;
	add: #SpTimestamp;
	add: #SpTranscript;
	add: #SpWeakArray;
	yourself.

package methodNames
	add: #Compiler -> #evaluate:in:to:;
	add: #SequencedStream -> #throughAll:;
	add: #TimeStamp -> #asSpTimestamp;
	add: #TimeStamp -> #printRFC1123StringOn:;
	add: #TimeStamp -> #rfc1123String;
	add: 'Locale class' -> #englishUS;
	add: 'Locale class' -> #evaluate:usingLocale:;
	yourself.

package binaryGlobalNames: (Set new
	yourself).

package globalAliases: (Set new
	yourself).

package setPrerequisites: (IdentitySet new
	add: 'Object Arts\Dolphin\Base\Dolphin';
	add: 'Object Arts\Dolphin\System\Base64\Dolphin Base64';
	add: 'Object Arts\Dolphin\MVP\Base\Dolphin MVP Base';
	add: 'Object Arts\Dolphin\Sockets\Dolphin Sockets';
	add: 'Object Arts\Dolphin\Sockets\Sockets Connection';
	yourself).

package!

"Class Definitions"!

Object subclass: #SpDate
	instanceVariableNames: 'underlyingDate'
	classVariableNames: ''
	poolDictionaries: ''
	classInstanceVariableNames: ''!
Object subclass: #SpEnvironment
	instanceVariableNames: ''
	classVariableNames: 'ImageShutdownTaskBlocks ImageStartupTaskBlocks'
	poolDictionaries: ''
	classInstanceVariableNames: ''!
Object subclass: #SpExceptionContext
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	classInstanceVariableNames: ''!
Object subclass: #SpFilename
	instanceVariableNames: 'filename'
	classVariableNames: ''
	poolDictionaries: ''
	classInstanceVariableNames: ''!
Object subclass: #SpFileStream
	instanceVariableNames: 'underlyingStream filename'
	classVariableNames: ''
	poolDictionaries: ''
	classInstanceVariableNames: ''!
Object subclass: #SpSocket
	instanceVariableNames: 'communicationDomain socketType protocolNumber underlyingSocket'
	classVariableNames: ''
	poolDictionaries: ''
	classInstanceVariableNames: ''!
Object subclass: #SpSocketAddress
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	classInstanceVariableNames: ''!
Object subclass: #SpStringUtilities
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	classInstanceVariableNames: ''!
Object subclass: #SpTimestamp
	instanceVariableNames: 'underlyingTimestamp'
	classVariableNames: ''
	poolDictionaries: ''
	classInstanceVariableNames: ''!
Object subclass: #SpTranscript
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	classInstanceVariableNames: ''!
Object subclass: #SpWeakArray
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	classInstanceVariableNames: ''!
Error subclass: #SpAbstractError
	instanceVariableNames: 'parameter'
	classVariableNames: ''
	poolDictionaries: ''
	classInstanceVariableNames: ''!
SpAbstractError subclass: #SpError
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	classInstanceVariableNames: ''!
SpError subclass: #SpSocketError
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	classInstanceVariableNames: ''!
SpSocketAddress subclass: #SpIPAddress
	instanceVariableNames: 'hostAddress portNumber'
	classVariableNames: ''
	poolDictionaries: ''
	classInstanceVariableNames: ''!

"Global Aliases"!


"Loose Methods"!

!Compiler methodsFor!

evaluate: aString in: context to: anObject

	^SpEnvironment evaluate: aString receiver: anObject in: context! !
!Compiler categoriesFor: #evaluate:in:to:!public! !

!Locale class methodsFor!

englishUS
	"Answer the English (United States) system locale."

	^Locale lcid: 1033!

evaluate: aBlock usingLocale: aLocale
	"Evaluate aBlock setting aLocale as the receiver default locale,
	and restoring the previous default locale after the evaluation of aBlock."

	| previousLocale |
	[ 
		previousLocale := self default.
		UserDefault := aLocale. 
		aBlock value 
	] ensure: [ UserDefault := previousLocale ]! !
!Locale class categoriesFor: #englishUS!accessing!public! !
!Locale class categoriesFor: #evaluate:usingLocale:!accessing!public! !

!SequencedStream methodsFor!

throughAll: aString 
	"Answer a collection of elements starting with the next element accessed by the receiver,
	and up to, and inclusive of, the next element that is equal to anObject. Positions the
	stream after anObject if found. If anObject is not in the collection, answer the entire rest
	of the collection. If the receiver is at its end, answer an empty Collection."

	|  newStream prevPosition |
	#eamAdded.
	newStream := self contentsSpecies writeStream: 128.
	prevPosition := self position.
	prevPosition = 0 ifTrue: [prevPosition := 1].
	(self skipToAll: aString) 
		ifTrue: [self position > 1 ifTrue: [newStream nextPutAll: (self copyFrom: prevPosition to: self position)]]
		ifFalse: [newStream nextPutAll: (self copyFrom: prevPosition to: self size )] .
	^newStream contents
! !
!SequencedStream categoriesFor: #throughAll:!public! !

!TimeStamp methodsFor!

asSpTimestamp
	"Answer an instance of SpTimestamp built from receiver."

	^SpTimestamp fromTimestamp: self!

printRFC1123StringOn: stream
	" Private - Prints the receiver in RFC1123 format over stream.
	E.g. Sun, 06 Nov 1994 08:49:37 GMT "

	| lcEnglishUS |
	lcEnglishUS := Locale englishUS.
	stream
		nextPutAll: (lcEnglishUS nameOfDay: self date weekdayIndex abbrev: true);
		nextPutAll: ', '.
	self date printOn: stream locale: lcEnglishUS format: 'dd MMM yyyy'.
	stream space.
	self time printOn: stream format: 'HH:mm:ss'.
	stream nextPutAll: ' GMT'!

rfc1123String
	"Answer a string representing receiver in RFC1123 date and time format.
	E.g. Sun, 06 Nov 1994 08:49:37 GMT . "

	| stream |
	#swAdded. #eamAdded.
	stream := String writeStream: 32.
	self printRFC1123StringOn: stream.
	^stream contents! !
!TimeStamp categoriesFor: #asSpTimestamp!converting!public! !
!TimeStamp categoriesFor: #printRFC1123StringOn:!printing!private! !
!TimeStamp categoriesFor: #rfc1123String!printing!public! !

"End of package definition"!

"Source Globals"!

"Classes"!

SpDate guid: (GUID fromString: '{D7238FE6-09FF-45DF-A706-9B96E35D0818}')!
SpDate comment: ''!
!SpDate categoriesForClass!Unclassified! !
!SpDate methodsFor!

< anotherSpDate 
	"^a Boolean
	Answer true if anotherSpDate is less (i.e. earlier) than me."
	^self underlyingDate < anotherSpDate underlyingDate!

<= anotherOSkDate 
	"^a Boolean
	Answer true if anotherOSkDate is greater (i.e. later) than me."

	^self underlyingDate <= anotherOSkDate underlyingDate!

= anotherOSkDate 
	"^a Boolean
	Answer true if anotherOSkDate is equivalent to me."
	^self underlyingDate = anotherOSkDate underlyingDate!

> anotherOSkDate 
	"^a Boolean
	Answer true if anotherOSkDate is greater (i.e. later) than me."
	^self underlyingDate > anotherOSkDate underlyingDate!

>= anotherOSkDate 
	"^a Boolean
	Answer true if anotherOSkDate is greater (i.e. later) than me."

	^self underlyingDate >= anotherOSkDate underlyingDate!

addDays: anInteger 
	"^a SpDate
	I don't change the date I represent.  Rather, I create a new date which represents
	my date offset by anInteger days."
	^SpDate fromDays: self asDays + anInteger!

addYears: anInteger 
	"^an OSkDate
I don't change the date I represent.  Rather, I create a new date which represents my date offset by anInteger years."

	^SpDate onDate: (Date 
				newDay: self underlyingDate dayOfMonth
				monthNumber: self underlyingDate monthIndex
				year: self underlyingDate year + anInteger)!

asDays
	"^an Integer
	I return the integer number of days between January 1, 1901 and
	the date I represent. "

	^self underlyingDate asDays!

asISO8610String 
	|targetStream|
	targetStream := WriteStream on: String new.
	self asISO8610StringOn: targetStream.
	^targetStream contents!

asISO8610StringOn: aStream 	
	aStream
		nextPutAll: self underlyingDate year printString;
		nextPut: $-.
	self underlyingDate monthIndex < 10 ifTrue: [aStream nextPut: $0].
	aStream
		nextPutAll: self underlyingDate monthIndex printString;
		nextPut: $-.
	self underlyingDate dayOfMonth < 10 ifTrue: [aStream nextPut: $0].
	aStream nextPutAll: self underlyingDate dayOfMonth printString.
	^self!

day
	^self underlyingDate day!

daysInMonth
	^Date daysInMonthIndex: self monthIndex forYear: self year!

hash
	"^an Object"
	^self underlyingDate hash!

max: anSpDate 
	^self > anSpDate ifTrue: [self] ifFalse: [anSpDate]!

min: anSpDate 
	^self < anSpDate ifTrue: [self] ifFalse: [anSpDate]!

monthIndex
	^self underlyingDate monthIndex!

onDate: aDate 
	underlyingDate := aDate.
	^self!

printOn: aStream
	self underlyingDate printOn: aStream!

subtractDays: anInteger 
	"^a SpDate
	I don't change the date I represent.  Rather, I create a new date which represents
	my date offset by anInteger days."
	^SpDate fromDays: self asDays - anInteger!

underlyingDate
	^underlyingDate!

weekdayIndex
	"Sunday=1, ... , Saturday=7"
	"Note that in Dolphin Smalltalk this actually is:
	Monday =1 , ... , Sunday=7. Who knows the correct answer?."

	^self underlyingDate weekdayIndex
!

year
	^self underlyingDate year! !
!SpDate categoriesFor: #<!comparing!public! !
!SpDate categoriesFor: #<=!comparing!public! !
!SpDate categoriesFor: #=!comparing!public! !
!SpDate categoriesFor: #>!comparing!public! !
!SpDate categoriesFor: #>=!comparing!public! !
!SpDate categoriesFor: #addDays:!public!services! !
!SpDate categoriesFor: #addYears:!public!services! !
!SpDate categoriesFor: #asDays!converting!public! !
!SpDate categoriesFor: #asISO8610String!printing!public! !
!SpDate categoriesFor: #asISO8610StringOn:!printing!public! !
!SpDate categoriesFor: #day!accessing!public! !
!SpDate categoriesFor: #daysInMonth!accessing!public! !
!SpDate categoriesFor: #hash!comparing!public! !
!SpDate categoriesFor: #max:!comparing!public! !
!SpDate categoriesFor: #min:!comparing!public! !
!SpDate categoriesFor: #monthIndex!accessing!public! !
!SpDate categoriesFor: #onDate:!initialize-release!public! !
!SpDate categoriesFor: #printOn:!printing!public! !
!SpDate categoriesFor: #subtractDays:!public!services! !
!SpDate categoriesFor: #underlyingDate!private! !
!SpDate categoriesFor: #weekdayIndex!accessing!public! !
!SpDate categoriesFor: #year!accessing!public! !

!SpDate class methodsFor!

fromDays: anInteger
	^self new onDate: (Date fromDays: anInteger)!

fromISO8610Stream: aStream 	
	| date |
	date := SpExceptionContext
		for: [self parseDateFromISO8601Stream: aStream]
		on: SpError
		do: [:ex | nil].
	^date isNil ifTrue: [nil] ifFalse: [self onDate: date]!

fromISO8610String: aString
	^aString size == 10
		ifFalse: [nil]
		ifTrue: [self fromISO8610Stream: aString readStream]!

integerOfLength: aLength FromString: aString
	"^an Integer or nil
	I parse an integer from aString, if I have problems I return nil.  I make sure
	the string form of the integer is exactly aLength characters long."
	"SpDate integerOfLength: 4 FromString: '2004'"
	^(aString size == aLength and: 
		[(aString asOrderedCollection select: [:aDigit | aDigit isDigit not]) isEmpty])
			ifFalse: [nil]
			ifTrue: [aString asNumber]!

newDay: day monthNumber: month year: year 
	^self new onDate: (Date newDay: day monthNumber: month year: year) !

onDate: aDate
	^self new onDate: aDate!

parseDateFromISO8601Stream: sourceStream	
	"^a Date or nil
	I parse an ISO 8601 date from sourceStream.  If there are any parsing
	problems, I return nil."
	| yyyy mm dd |
	yyyy := self integerOfLength: 4 FromString: (sourceStream upTo: $-).
	mm := self integerOfLength: 2 FromString: (sourceStream upTo: $-).
	dd := self integerOfLength: 2 FromString: (sourceStream next: 2).
	(yyyy isNil or: [mm isNil or: [dd isNil]]) ifTrue: [^nil].
	^SpExceptionContext 
		for: [Date newDay: dd monthNumber: mm year: yyyy]
		on: SpError
		do: [:ex | nil]!

today
	^self onDate: Date today! !
!SpDate class categoriesFor: #fromDays:!instance creation!public! !
!SpDate class categoriesFor: #fromISO8610Stream:!instance creation!public! !
!SpDate class categoriesFor: #fromISO8610String:!instance creation!public! !
!SpDate class categoriesFor: #integerOfLength:FromString:!private! !
!SpDate class categoriesFor: #newDay:monthNumber:year:!instance creation!public! !
!SpDate class categoriesFor: #onDate:!instance creation!public! !
!SpDate class categoriesFor: #parseDateFromISO8601Stream:!private! !
!SpDate class categoriesFor: #today!instance creation!public! !

SpEnvironment guid: (GUID fromString: '{680E304E-A02E-4C24-B34B-A2A1C9025422}')!
SpEnvironment comment: 'Prueba'!
!SpEnvironment categoriesForClass!Unclassified! !
!SpEnvironment class methodsFor!

addImageShutdownTask: aBlock for: anObject 
	"^self
	I add aBlock to the list of actions and note that this is for anObject"

	self imageShutdownTaskBlocks at: anObject put: aBlock.
	^self
!

addImageStartupTask: aBlock for: anObject 
	"^self
	I add aBlock to the list of actions and note that this is for anObject"

	self imageStartupTaskBlocks at: anObject put: aBlock.
	^self
!

allSubclassesOf: aClass
	"^an Array
	I return the array of classes which are subclasses of aClass."

	^aClass allSubclasses asArray
	
"SpEnvironment allSubclassesOf: Error"!

base64DecodeAsString: base64EncodedString
	"Decodes a base64EncodedString, and answer the string decoded. "
 
	| outputStream |
	#eamAdded.
	outputStream := ByteArray writeStream: base64EncodedString size.
	Base64Codec decodeFrom: base64EncodedString readStream onto: outputStream.
	^outputStream contents asString!

base64EncodeString: aString 

	| outputStream |
	#eamAdded.
	outputStream := WriteStream on: String new.
	Base64Codec encodeFrom:  aString asByteArray readStream onto: outputStream.
	^outputStream contents asString!

byteArrayFromHexString: aString 
	
	^ByteArray fromHexString: aString!

characterFromInteger: anInteger 
	^Character value: anInteger
	
"SpEnvironment characterFromInteger: 32"!

collectGarbage

	#eamAdded.
	MemoryManager current collectGarbage!

compactingGC

	#eamAdded.
	MemoryManager current compact!

evaluate: aString in: anEnvironment

	^Compiler evaluate: aString!

evaluate: aString receiver: anObject in: anEnvironment 

	#eamChanged.
	^Compiler evaluate: aString for: anObject logged: false
!

hexStringFromByteArray: aByteArray 

	^String streamContents: [:stream | aByteArray printHexOn: stream]!

imageShutdownTaskBlocks
	"self imageShutdownTaskBlocks"
	ImageShutdownTaskBlocks isNil 
		ifTrue: [ImageShutdownTaskBlocks := WeakIdentityDictionary new].
	^ImageShutdownTaskBlocks!

imageStartupTaskBlocks
	"self imageStartupTaskBlocks"
	ImageStartupTaskBlocks isNil 
		ifTrue: [ImageStartupTaskBlocks := WeakIdentityDictionary new].
	^ImageStartupTaskBlocks!

initialize
	super initialize.
	SessionManager current
		when: #sessionStopped
			send: #shutDown
			to: self;
		when: #sessionStarted
			send: #startUp
			to: self!

integerFromString: aString
	"^an Integer
	We need this because of what looks like a bug in GemStone's String>>asNumber 
	(e.g. '16rFF' -> 1.6000000000000000E+01, not 255)."

	^aString asNumber
	
"SpEnvironment integerFromString: '32' "!

isAmbraiSmalltalk
	^false!

isDolphin
	^true!

isGemStone
	^false!

isGNUSmalltalk
	^false!

isHeadless
	^RuntimeSessionManager current isConsoleApplication!

isObjectStudio
	^false!

isSmalltalkX
	^false!

isSmalltalkXY
	^false!

isSqueak
	^false!

isVASmalltalk
	^false!

isVisualSmalltalk
	^false!

isVisualWorks
	^false!

onUnix
	"we are running on Unix, yes or no?"
	^false!

onWindows
	"we are running on Windows, yes or no?"
	^true!

removeShutdownActionFor: anObject 
	"^self
	I remove the task block for an object it it has one"

	self imageShutdownTaskBlocks removeKey: anObject ifAbsent: [nil].
	^self!

removeStartupActionFor: anObject 
	"^self
	I remove the task block for an object it it has one"

	self imageStartupTaskBlocks removeKey: anObject ifAbsent: [nil].
	^self!

runShellCommandString: aCommandString

" not yet ported!!
 
	See ShellLibrary
"!

shutDown
	"Squeak specific!!"
	self imageShutdownTaskBlocks values do: [:shutdownTask | shutdownTask value].
	^self!

startUp
	"Squeak specific!!"
	self imageStartupTaskBlocks values do: [:startupTask | startupTask value].
	^self!

streamStartPosition
	"^an Integer
	Streams start at position 0 in VisualWorks & Squeak, and position 1 in
	GemStone(!!). "
	^ 0!

uninitialize

	SessionManager current removeEventsTriggeredFor: self!

writeStackDumpForException: exception to: targetStream 
	
	exception printTraceOn: targetStream! !
!SpEnvironment class categoriesFor: #addImageShutdownTask:for:!image startup/shutdown!public! !
!SpEnvironment class categoriesFor: #addImageStartupTask:for:!image startup/shutdown!public! !
!SpEnvironment class categoriesFor: #allSubclassesOf:!public!queries! !
!SpEnvironment class categoriesFor: #base64DecodeAsString:!development!public! !
!SpEnvironment class categoriesFor: #base64EncodeString:!public! !
!SpEnvironment class categoriesFor: #byteArrayFromHexString:!hex!public! !
!SpEnvironment class categoriesFor: #characterFromInteger:!public!services! !
!SpEnvironment class categoriesFor: #collectGarbage!development!public! !
!SpEnvironment class categoriesFor: #compactingGC!development!public! !
!SpEnvironment class categoriesFor: #evaluate:in:!public! !
!SpEnvironment class categoriesFor: #evaluate:receiver:in:!public! !
!SpEnvironment class categoriesFor: #hexStringFromByteArray:!hex!public! !
!SpEnvironment class categoriesFor: #imageShutdownTaskBlocks!private! !
!SpEnvironment class categoriesFor: #imageStartupTaskBlocks!private! !
!SpEnvironment class categoriesFor: #initialize!public!testing! !
!SpEnvironment class categoriesFor: #integerFromString:!public!services! !
!SpEnvironment class categoriesFor: #isAmbraiSmalltalk!public!testing-dialects! !
!SpEnvironment class categoriesFor: #isDolphin!public!testing-dialects! !
!SpEnvironment class categoriesFor: #isGemStone!public!testing-dialects! !
!SpEnvironment class categoriesFor: #isGNUSmalltalk!public!testing-dialects! !
!SpEnvironment class categoriesFor: #isHeadless!public! !
!SpEnvironment class categoriesFor: #isObjectStudio!public!testing-dialects! !
!SpEnvironment class categoriesFor: #isSmalltalkX!public!testing-dialects! !
!SpEnvironment class categoriesFor: #isSmalltalkXY!public!testing-dialects! !
!SpEnvironment class categoriesFor: #isSqueak!public!testing-dialects! !
!SpEnvironment class categoriesFor: #isVASmalltalk!public!testing-dialects! !
!SpEnvironment class categoriesFor: #isVisualSmalltalk!public!testing-dialects! !
!SpEnvironment class categoriesFor: #isVisualWorks!public!testing-dialects! !
!SpEnvironment class categoriesFor: #onUnix!public!testing! !
!SpEnvironment class categoriesFor: #onWindows!public!testing! !
!SpEnvironment class categoriesFor: #removeShutdownActionFor:!image startup/shutdown!public! !
!SpEnvironment class categoriesFor: #removeStartupActionFor:!image startup/shutdown!public! !
!SpEnvironment class categoriesFor: #runShellCommandString:!os commands!public! !
!SpEnvironment class categoriesFor: #shutDown!private! !
!SpEnvironment class categoriesFor: #startUp!private! !
!SpEnvironment class categoriesFor: #streamStartPosition!public!services! !
!SpEnvironment class categoriesFor: #uninitialize!public! !
!SpEnvironment class categoriesFor: #writeStackDumpForException:to:!public!services! !

SpExceptionContext guid: (GUID fromString: '{81B0403A-1FBA-4761-B4BF-8C3D14CA67C3}')!
SpExceptionContext comment: 'Exceptions vary quite a bit between Smalltalk implementaions, despite the presence of the ANSI Smalltalk specification.  This class representss a portable exception context in which a block can be executed, exceptions trapped and handlers defined.'!
!SpExceptionContext categoriesForClass!Unclassified! !
!SpExceptionContext class methodsFor!

brokenPipeException
	"I return the exception that get's thrown when a socket connection gets 
	broken."

	^SocketError!

for: aBlock on: anException do: exceptionBlock
	"^an Object
	I return the result of evaluating aBlock. In VisualWorks and other
	Smalltalks which are ANSI compliant, I delegate to aBlock."

	^aBlock on: anException do: exceptionBlock!

for: aBlock onAnyExceptionDo: exceptionBlock 
	"^an Object 
	I execute aBlock and if there is any exception I evaluate exceptionBlock.  Essentially, I look out for the most abstract kind of exception which , of course, will vary between Smalltalk implementations."

	#eamAdded.
	^aBlock on: self generalException do: exceptionBlock!

generalException
	"I return the exception that is the most general in this dialect."

	#eamAdded.
	^Error! !
!SpExceptionContext class categoriesFor: #brokenPipeException!native exceptions!private! !
!SpExceptionContext class categoriesFor: #for:on:do:!instance creation!public! !
!SpExceptionContext class categoriesFor: #for:onAnyExceptionDo:!instance creation!private! !
!SpExceptionContext class categoriesFor: #generalException!native exceptions!private! !

SpFilename guid: (GUID fromString: '{5C58AF74-114B-4489-B793-88CAAD265AC5}')!
SpFilename comment: '
A SpFilename represents a file or directory and allows operations like delete, makeDirectory, etc
Part of code is copied with permission from Cincom VisualWorks

Instance Variables
	filename:		<String> name of a file or directory

'!
!SpFilename categoriesForClass!Unclassified! !
!SpFilename methodsFor!

appendStream
	"^a SpFileStream
	I create an append stream on the file I represent."

	^SpFileStream appendingToFilename: self!

asAbsoluteFilename
	"Answer a Filename pointing to the same file using absolute path.
	The method may answer the receiver it it is already absolute."

	^self isAbsolute 
		ifTrue: [self]
		ifFalse: [self class named: (ImageRelativeFileLocator current localFileSpecFor: self asString)]!

asFilename
	^self!

asString
	^self filename!

construct: extraFn
	"Make a new instance, treating the receiver as a directory, and
	the string argument as a file within the pathname."

	^self class named: self filename , self separator asString , extraFn!

contentsOfEntireFile
	| stream |
	[stream := FileStream read: self filename.
	^stream contents]
		ensure: [stream close]
!

createdTimestamp
	"a SpTimestamp
	timestamp of file creation. "

	^SpTimestamp fromTimeStamp: self fileEntry ftCreationTime asSYSTEMTIME asTimeStamp
!

delete
	self isDirectory
		ifTrue: [File deleteDirectory: self path]
		ifFalse: [File delete: self path]!

directory
	"a filename of the directory for this Filename."
	^self class named: self head!

directoryContents

	| contents |
	contents := OrderedCollection new.
	File for: '*.*' in: self head do: [:each | contents add: (self class named: each path)].
	^contents
	!

exists	
	"^a Boolean
	I return true if the file or direcotory I represent actually exists"
	
	^File exists: self path!

extension
	"Answer the receiver's extension if any.  This is the characters from the
	 last occurrence of a period to the end, inclusive. E.g. the extension of
	 'squeak.image' is '.image'. Answer nil if none.  Note that e.g. .login has no
	 extension."

	| string periodIndex |
	string := self tail.
	periodIndex := string lastIndexOf: $..
	^periodIndex > 1 ifTrue: [string copyFrom: periodIndex to: string size]!

fileEntry
	"Private - Answer the WIN32_FIND_DATA."

	| entries |
	entries := File find: self path.
	entries isEmpty ifTrue: [^SpError raiseSignal: 'File not found'].
	^entries first
!

filename
	 "^a String"
	^filename!

fileSize
	
	^self fileEntry fileSize
!

head
	"Answer the directory prefix as a String."

	^File splitPathFrom: self path!

isAbsolute
	"Answer true if this name is absolute (e.g. not relative to the
	'current directory')."

	self asString size = 0 ifTrue: [^false].
	^(self asString at: 1) == self separator!

isDirectory
	
	^self fileEntry isDirectory!

isRelative
	"Answer true if this name must be interpreted relative to some directory."
	^self isAbsolute not!

lastSeparatorIndex
	^self asString lastIndexOf: self separator ifAbsent: [nil]!

makeDirectory
	
	^File createDirectory: self path!

modifiedTimestamp
	"a SpTimestamp
	timestamp of last file modification"

	| ts |
	ts := self fileEntry ftLastWriteTime asSYSTEMTIME asTimeStamp.
	^SpTimestamp
		fromDate: ts date
		andTime: (Time fromSeconds: ts time asSeconds) "Truncation of the millisecond precision"
!

named: aString 
	"^self
	I initialize myself to represent the filename identified by aString."

	filename := aString.
	^self!

path
	"Answer a path String representing the AbsoluteFilename string of the receiver."

	^self asAbsoluteFilename asString!

readStream
	"^a SpFileStream
	I create a read stream on the file I represent."

	^SpFileStream readingFromFilename: self!

separator
	"Answer the platform's filename component separator."

	^File pathDelimiter!

tail
	"Answer the filename suffix as a String."

	^File splitFilenameFrom: self path!

writeStream
	"^a SpFileStream
	I create a write stream on the file I represent."

	^SpFileStream writingToFilename: self! !
!SpFilename categoriesFor: #appendStream!public!services! !
!SpFilename categoriesFor: #asAbsoluteFilename!accessing!public! !
!SpFilename categoriesFor: #asFilename!accessing!public! !
!SpFilename categoriesFor: #asString!accessing!public! !
!SpFilename categoriesFor: #construct:!private! !
!SpFilename categoriesFor: #contentsOfEntireFile!public!services! !
!SpFilename categoriesFor: #createdTimestamp!accessing!public! !
!SpFilename categoriesFor: #delete!public!services! !
!SpFilename categoriesFor: #directory!accessing!public! !
!SpFilename categoriesFor: #directoryContents!public!services! !
!SpFilename categoriesFor: #exists!public!testing! !
!SpFilename categoriesFor: #extension!accessing!public! !
!SpFilename categoriesFor: #fileEntry!accessing!private! !
!SpFilename categoriesFor: #filename!private! !
!SpFilename categoriesFor: #fileSize!accessing!public! !
!SpFilename categoriesFor: #head!accessing!public! !
!SpFilename categoriesFor: #isAbsolute!public!testing! !
!SpFilename categoriesFor: #isDirectory!public!testing! !
!SpFilename categoriesFor: #isRelative!public!testing! !
!SpFilename categoriesFor: #lastSeparatorIndex!private! !
!SpFilename categoriesFor: #makeDirectory!public!services! !
!SpFilename categoriesFor: #modifiedTimestamp!accessing!public! !
!SpFilename categoriesFor: #named:!initialize-release!public! !
!SpFilename categoriesFor: #path!accessing!public! !
!SpFilename categoriesFor: #readStream!public!services! !
!SpFilename categoriesFor: #separator!private! !
!SpFilename categoriesFor: #tail!accessing!public! !
!SpFilename categoriesFor: #writeStream!public!services! !

!SpFilename class methodsFor!

named: aString 
	"^a SpFilename
	I create a new instance of myself to represent the filename identified by aString."

	^self new named: aString! !
!SpFilename class categoriesFor: #named:!instance creation!public! !

SpFileStream guid: (GUID fromString: '{3887F6DC-2F8A-4180-9844-3477BAA7C1A4}')!
SpFileStream comment: ''!
!SpFileStream categoriesForClass!Unclassified! !
!SpFileStream methodsFor!

appendingToFilename: aSpFilename
	self filename: aSpFilename.
	underlyingStream := FileStream write: self filename path mode: #append!

atEnd
	^self underlyingStream atEnd!

binary
	^self underlyingStream isBinary!

close
	^self underlyingStream close!

closed
	^self underlyingStream file isNil!

contentsStream
	^self underlyingStream contents readStream!

cr
	^self underlyingStream cr!

filename
	^filename!

filename: aSpFilename
	filename := aSpFilename!

flush
	^self underlyingStream flush!

next
	^self underlyingStream next!

nextPut: anObject 
	^self underlyingStream nextPut: anObject!

nextPutAll: aCollection 
	^self underlyingStream nextPutAll: aCollection!

peek
	^self underlyingStream peek!

position
	^self underlyingStream position!

position: aNumber
	^self underlyingStream position: aNumber!

readingFromFilename: anOSkFilename 
	^self readingFromFileNamed: anOSkFilename asString.!

readingFromFileNamed: aString 
	"^self
I initialize myself to write to a file named aString."

	filename := aString.
	underlyingStream := FileStream read: aString text: false.
	^self!

skip: anInteger
	^self underlyingStream skip: anInteger!

store: anObject 
	^self underlyingStream store: anObject!

throughAll: aCollection
	^self underlyingStream throughAll: aCollection!

underlyingStream
	^underlyingStream!

upTo: anObject
	^self underlyingStream upTo: anObject!

upToAll: aCollection
	^self underlyingStream upToAll: aCollection!

upToEnd
	^self underlyingStream upToEnd!

writingToFilename: aSpFilename 
	self filename: aSpFilename.
	underlyingStream := FileStream write: self filename path.
	^self!

writingToFileNamed: aString 
	"^self
I initialize myself to write to a file named aString."

	#eamAdded.
	filename := aString.
	underlyingStream := FileStream write: aString.
	^self! !
!SpFileStream categoriesFor: #appendingToFilename:!initialize-release!public! !
!SpFileStream categoriesFor: #atEnd!public!services! !
!SpFileStream categoriesFor: #binary!public!services! !
!SpFileStream categoriesFor: #close!public!services! !
!SpFileStream categoriesFor: #closed!public!services! !
!SpFileStream categoriesFor: #contentsStream!accessing!public! !
!SpFileStream categoriesFor: #cr!public!services! !
!SpFileStream categoriesFor: #filename!accessing!public! !
!SpFileStream categoriesFor: #filename:!accessing!public! !
!SpFileStream categoriesFor: #flush!public!services! !
!SpFileStream categoriesFor: #next!public!services! !
!SpFileStream categoriesFor: #nextPut:!public!services! !
!SpFileStream categoriesFor: #nextPutAll:!public!services! !
!SpFileStream categoriesFor: #peek!public!services! !
!SpFileStream categoriesFor: #position!public!services! !
!SpFileStream categoriesFor: #position:!public!services! !
!SpFileStream categoriesFor: #readingFromFilename:!initialize-release!private! !
!SpFileStream categoriesFor: #readingFromFileNamed:!initialize-release!private! !
!SpFileStream categoriesFor: #skip:!public!services! !
!SpFileStream categoriesFor: #store:!public!services! !
!SpFileStream categoriesFor: #throughAll:!public!services! !
!SpFileStream categoriesFor: #underlyingStream!accessing!public! !
!SpFileStream categoriesFor: #upTo:!public!services! !
!SpFileStream categoriesFor: #upToAll:!public!services! !
!SpFileStream categoriesFor: #upToEnd!public!services! !
!SpFileStream categoriesFor: #writingToFilename:!initialize-release!public! !
!SpFileStream categoriesFor: #writingToFileNamed:!initialize-release!private! !

!SpFileStream class methodsFor!

appendingToFilename: aSpFilename 
	"^a SpFileStream
	I create a new instance of myself to append to the file identified by anOSkFilename."

	^self new appendingToFilename: aSpFilename!

readingFromFilename: aSpFilename 
	"^a SpFileStream
	I create a new instance of myself to read the file identified by anOSkFilename."

	^self new readingFromFilename: aSpFilename!

readingFromFileNamed: aString 
	"^a SpFileStream
	I create a new instance of myself to read from a file named aString."

	^self new readingFromFileNamed: aString!

writingToFilename: aSpFilename 
	"^an SpFileStream
	I create a new instance of myself to append to the file identified by aSpFilename."

	^self new writingToFilename: aSpFilename!

writingToFileNamed: aString 
	"^a SpFileStream
	I create a new instance of myself to write to a file named aString."

	^self new writingToFileNamed: aString! !
!SpFileStream class categoriesFor: #appendingToFilename:!instance creation!public! !
!SpFileStream class categoriesFor: #readingFromFilename:!instance creation!public! !
!SpFileStream class categoriesFor: #readingFromFileNamed:!instance creation!public! !
!SpFileStream class categoriesFor: #writingToFilename:!instance creation!public! !
!SpFileStream class categoriesFor: #writingToFileNamed:!instance creation!public! !

SpSocket guid: (GUID fromString: '{CB2D39BE-1FCC-43AD-A330-ECF9FA634D16}')!
SpSocket comment: 'OSkSocket  instances represent BSD sockets.  The idea here is to stick as closely to the BSD (Posix) spec terminology as possible.  I''ve cheated and taken the Linux man pages as being a good description of BSD sockets, and all the names etc are taken from the linux man pages (start with man socket).

The key steps in using a BSD socket for a server is:
	o int socket(int domain, int type, int protocol) [see: man socket]
		- create a new instance of a socket  where domain is the protocol
		 family (see below), type is the communication semantics (see below)
		 and protocol is a specific protocol (though there is usually only one
		 protocol for a given domain/type combination).
		 See: OSkSocket class>>forDomain:type:protocol:
	o  int  bind(int  sockfd, struct sockaddr *my_addr, socklen_t addrlen) [see: man bind]
		- let the socket know what address it is representing.  sockfd is the id of 
		 the socket, sockaddr is the address to be represented and addrlen is 
		 the length in bytes of the address. 
		 See OSkSocket>>bindSocketAddress: 
	o int listen(int s, int backlog) [see: man listen]
		- have the socket s listen for inbound requests, and allow up to backlog
		 requests to be queued at any one time.
		 See OSkSocket>>listenBackloggingUpTo: 
	o  int   accept(int   s,  struct  sockaddr  *addr,  socklen_t *addrlen) [see: man accept]
		- accept the next inbound request from the backlog.  Accept the request from
		 socket s, and spawn a new socket to handle the request (freeing s to accept
		 the next accept).  The new socket if defined in sockaddr which has a length of
		 addrlen..
		 See OSkSocket>>accept

For sockets connecting to existing server sockets, the steps are"
	o socket() [as above]
	o  int  connect(int sockfd, const struct sockaddr *serv_addr, socklen_t addrlen)
		- Have socket sockfd connect to the remote socket at serv_addr (where serv_addr
		 is addrlen bytes long).  
		 See OSkSocket>>connectTo:

A socket appears in many ways to be a file descriptor, and the regular read and write operations can be used once a socket is open.  Accordingly, some of the file control (man fcntl) options can be used.  Of particular interest is the idea of blocking and non-blocking operation.

Blocking is the default mode for an active socket.  This means that (for example) when reading from a socket the entire process(!!) blocks waiting for the read operation to complete.  In non-blocking mode the same read operation would return immediately, but might not return anything useful.

See man accept for more information on blocking/non-blocking.

So, roughly speaking, blocking is easier to program for because you code as if the read always gets something.  A blocking interface is harder because you have to check if you got something, and try again later if you didn''t - but non-blocking lets the process get on with other things while your thread is waiting, and this can be a Very Good Thing.

Now, VisualWorks seems to interpret blocking to be per Smalltalk Process.  To make this work, the VisualWorks socket implementation always asks the operating system for non-blocking sockets, but presents these as approproate (blocking or non-blocking) to the Smalltalk Process.  This means that work can continue in a VisualWorks VM even when a socket is being used in (apparantly) blocking mode.

A GemStone Gem asks the operating system for the kind of socket asked for by the Process.  So, if a request is made for a blocking socket, the entire Gem blocks on reads from that socket. 

Just beware the difference between blocking and non-blocking sockets, and also of the differing implemenmtations between Smalltalk dialects.


'!
!SpSocket categoriesForClass!Unclassified! !
!SpSocket methodsFor!

accept
	"^an OSkSocket
	I accept the next connection made to the server socket I represent.  I return a new
	instance of OSkSocket which represents the socket over which information can be
	exchanged.
	NOTE: this call will block waiting for an inbound connection"

	^SpExceptionContext 
		for: [self class onNativeclientSocket: self underlyingSocket accept for: self]
		on: Error
		do: [:ex | SpSocketError signalWith: ex]!

acceptRetryingIfTransientErrors
	"I try to do an accept.  If I get an exception which is 'transient' I retry.
	Note: Added SocketWouldBlock, but I really don't know if exists 
	some type of 'transient error' in Dolphin."

	#eamChanged.
	^SpExceptionContext 
		for: [self accept]
		on: SocketWouldBlock 
		do: [:ex | ex retry]!

bindSocketAddress: aSocketAddress 
	"^self
	Equivalent of:  bind(int  sockfd, struct sockaddr *my_addr, socklen_t addrlen);
	see man bind.
	Bind the socket to aSocketAddress."

	self underlyingSocket bindPort: aSocketAddress portNumber.
	^self!

close
	"^self
	The same as the close() posix function."

	self underlyingSocket close.
	^self!

communicationDomain
	^communicationDomain!

connectTo: aSocketAddress 
	"^self
	I instruct my underlying socket to connect to aSocketAddress."

	self underlyingSocket port: aSocketAddress portNumber address: aSocketAddress hostAddress.
	self underlyingSocket  connect.
	^self!

getPeerName
	"^an OSkSocketAddress
	see man getpeername.
	I return the socket address of the other/remote/peer end of the socket I represent."

	#eamChanged.
	^SpIPAddress on: self underlyingSocket peerAddress for: self!

getSocketName
	"^an OSkSocketAddress
	see: man getsockname
	I return my local socket address which may be any subclass of 
	OSkSocketAddress."

	^SpSocketAddress on: self underlyingSocket address for: self!

isActive
	"Answer whether receiver is active or not."

	#eamAdded.
	^self underlyingSocket isOpen!

listenBackloggingUpTo: aNumberOfConnections 
	"I set the socket I represent listening for incomming connections, allowing a 
	backlog of up to aNumberOfConnections."

	#eamAdded.
	self underlyingSocket listen: aNumberOfConnections.
	^self!

onNativeclientSocket: aSocket for: aServerSocket 
	"I initialize myself with the same properties as aServerSocket and with
	aNativeSocket as my underlying socket."

	#eamAdded.
	underlyingSocket := aSocket.
	^self!

onNativeSocket: aSocket forDomain: aCommunicationDomain type: aSocketType protocol: aProtocolNumber 
	"^self
	I initialize myself as a socket defined by the communications domain, type and protocol.  This
	follows the equivalent of the using the socket()  or socketpair() function."

	communicationDomain := aCommunicationDomain.
	socketType := aSocketType.
	protocolNumber := aProtocolNumber.
	underlyingSocket := aSocket.
	^self!

protocolNumber
	^protocolNumber!

read: targetNumberOfBytes 
	"I attempt to read the targetNumberOfBytes from my underlying socket.  If the targetNumberOfBytes
	are not available, I return what I can get."

	#eamAdded.
	^SpExceptionContext 
		for: [
			| buffer count |
			buffer := ByteArray new: targetNumberOfBytes.
			count := self underlyingSocket receiveSome: buffer count: buffer size startingAt: 1.
			buffer copy: ByteArray from: 1 to: count ]
		on: SocketError
		do: [:ex | SpSocketError signalWith: ex]!

readInto: aByteArray startingAt: startIndex for: aNumberOfBytes 
	"I return the number of bytes actually read."

	^self underlyingSocket receiveSome: aByteArray count: aNumberOfBytes startingAt: startIndex !

readStream
	"^nil
While this is convinient, it would mean that sockets would depend on streams, and we want to avoid that.
Use {my stream class} on: {my socket} instead."

	^self shouldNotImplement!

readyForRead
	"I return true if a read operation will return some number of bytes."

	#eamAdded.
	^self underlyingSocket hasInput!

send: aByteObject count: countInteger startingAt: startInteger 
	"Private - Reimplemented here to avoid override of Dolphin Smalltalk version."

	| sent |
	sent := 0.
	[sent < countInteger] whileTrue: 
			[sent := sent + (self underlyingSocket
								sendSome: aByteObject
								count: countInteger - sent
								startingAt: sent + startInteger)].
	^sent!

setAddressReuse: aBoolean 
	"If a boolean is true, I set address reuse on, otherwise I set address reuse off.
	Dolphin uses SOCK_READDR by default, so do nothing."

	#eamChanged!

setOptionForLevel: aLevelNumber optionID: anOptionNumber value: aValue 
	"^self
	see man 2 setsockopt
	From the manpage:
	Getsockopt  and  setsockopt  manipulate  the  options associated with a
       socket.  Options may exist at multiple protocol levels; they are always
       present at the uppermost socket level."

	self underlyingSocket 
		setOptionsLevel: aLevelNumber
		name: anOptionNumber
		value: aValue.
	^self!

setUnderlyingSocket: aSocket 
	"Private - Set aSocket as the underlying socket of receiver."

	#eamAdded.
	underlyingSocket := aSocket!

socketType
	^socketType!

underlyingSocket
	^underlyingSocket!

underlyingSocket: aSocket2
	"Private - Set  aSocket2 as the underlying socket of receiver." 

	underlyingSocket := aSocket2!

waitForReadDataUpToMs: millisecondCount 
	"Return true if data became available within millisecondCount, and
	false if timed out."
	"Sean M. said: this is dodgy. Esteban says: I don't know how to make it better."

	| deadline |
	deadline := TimeStamp current asMilliseconds + millisecondCount.
	[ TimeStamp current asMilliseconds > deadline ifTrue: [ ^false ]. self underlyingSocket hasInput ] whileFalse.
	^self underlyingSocket hasInput 
!

write: sourceByteArray 
	"^I write the contents of the sourceByteArray to my underlying Socket.
	I return the number of bytes written."

	#eamAdded.
	^SpExceptionContext 
		for: [self underlyingSocket sendSome: sourceByteArray count: sourceByteArray size startingAt: 1]
		on: Error
		do: [:ex | SpSocketError signalWith: ex]!

writeFrom: aByteArray startingAt: startIndex for: length 
	"I return the number of bytes actually written."

	^SpExceptionContext
		for: [self send: aByteArray
				count: length
				startingAt: startIndex]
		on: Error
		do: [:ex | SpSocketError raiseSignal: ex]!

writeStream
	"^nil
While this is convinient, it would mean that sockets would depend on streams, and we want to avoid that.
Use {my stream class} on: {my socket} instead."

	^self shouldNotImplement! !
!SpSocket categoriesFor: #accept!private!services-status! !
!SpSocket categoriesFor: #acceptRetryingIfTransientErrors!public! !
!SpSocket categoriesFor: #bindSocketAddress:!private!services-status! !
!SpSocket categoriesFor: #close!public!services-status! !
!SpSocket categoriesFor: #communicationDomain!accessing!private! !
!SpSocket categoriesFor: #connectTo:!private!services-status! !
!SpSocket categoriesFor: #getPeerName!private!services-accessing! !
!SpSocket categoriesFor: #getSocketName!private!services-accessing! !
!SpSocket categoriesFor: #isActive!public!testing! !
!SpSocket categoriesFor: #listenBackloggingUpTo:!public!services-status! !
!SpSocket categoriesFor: #onNativeclientSocket:for:!private! !
!SpSocket categoriesFor: #onNativeSocket:forDomain:type:protocol:!initialize-release!private! !
!SpSocket categoriesFor: #protocolNumber!accessing!private! !
!SpSocket categoriesFor: #read:!public!services-io! !
!SpSocket categoriesFor: #readInto:startingAt:for:!public!services-io! !
!SpSocket categoriesFor: #readStream!private!services-streams! !
!SpSocket categoriesFor: #readyForRead!public!services-io!testing! !
!SpSocket categoriesFor: #send:count:startingAt:!private!services-io! !
!SpSocket categoriesFor: #setAddressReuse:!private!services-options! !
!SpSocket categoriesFor: #setOptionForLevel:optionID:value:!private!services-status! !
!SpSocket categoriesFor: #setUnderlyingSocket:!private! !
!SpSocket categoriesFor: #socketType!accessing!private! !
!SpSocket categoriesFor: #underlyingSocket!private! !
!SpSocket categoriesFor: #underlyingSocket:!private! !
!SpSocket categoriesFor: #waitForReadDataUpToMs:!private!services-io! !
!SpSocket categoriesFor: #write:!public!services-io! !
!SpSocket categoriesFor: #writeFrom:startingAt:for:!public!services-io! !
!SpSocket categoriesFor: #writeStream!private!services-streams! !

!SpSocket class methodsFor!

clientSocketClass
	"Private - Answer the class used as underlying socket in the instances of receiver created as client sockets."

	#eamAdded.
	^Socket2!

connectToServerOnHost: hostName port: portNumber 
	"^an OSkSocket 
	I return a new instance of myself which represents a socket connecter to a server listening on portNumber at hostName."

	| newSocket |
	newSocket := self newTCPSocket.
	newSocket connectTo: (SpIPAddress hostName: hostName port: portNumber).
	^newSocket!

newSocketPair
	"Answer an array of two elements containing a pair of sockets connected between them."

	"NOTE: This implementation was taken from Sean Malloy's #connectedPair. "

	| listener server client port |
	port := (((Random new next: 100) detect: [ :ea | ea > 0.11 ] ifNone: [ 0.21 ]) * 10000) asInteger.
	[ listener := self port: port backlog: 4.	server := listener accept. listener close ] fork.
	(Delay forMilliseconds: 100) wait.
	client := self connectToServerOnHost: 'localhost' port: port.
	(Delay forMilliseconds: 100) wait.
	server ifNil: [self error: 'No server'].
	^Array with: server with: client
!

newTCPClientSocket
	"Answer a new instance of a TCP socket, created to be client (i.e. it will request the connection)."

	#eamAdded.
	^(self new)
		setUnderlyingSocket: self clientSocketClass basicNew;
		yourself!

newTCPServerSocket
	"Answer a new instance of receiver configured to be a TCP Server socket."

	#eamAdded.
	^(self new)
		setUnderlyingSocket: self serverSocketClass new;
		yourself!

newTCPSocket
	"Answer a new instance of receiver configured to be a TCPSocket. By default, instantiates a client TCPSocket." 

	#eamAdded.
	^self newTCPClientSocket!

on: aSocket2 
	"Answer a new instance of receiver with aSocket2 as its underlying socket."

	^self new underlyingSocket: aSocket2!

onNativeclientSocket: aNativeSocket for: aServerSocket 
	"^an OSkSocket
I create a new instance of my self at the request of aServerSocket  where
this new instance will be a connected client socket (connected via aNativeSoket)."

	^self new onNativeclientSocket: aNativeSocket for: aServerSocket!

port: portNumber 
	"Answer a new instance of receiver, configured to be a server socket serving on portNumber."

	#eamAdded.
	^(self new)
		setUnderlyingSocket: (self serverSocketClass port: portNumber);
		yourself!

port: portNumber backlog: anInteger 
	"Answer a new instance of receiver, configured to a server socket
	serving on portNumber, with a backlog of anInteger connections."

	#eamAdded.
	^(self new)
		setUnderlyingSocket: (ServerSocket2 port: portNumber backlog: anInteger);
		yourself!

serverSocketClass
	"Private - Answer the class used as underlying socket in the instances of receiver created as server sockets."

	#eamAdded.
	^ServerSocket2! !
!SpSocket class categoriesFor: #clientSocketClass!constants!private!services! !
!SpSocket class categoriesFor: #connectToServerOnHost:port:!public!services! !
!SpSocket class categoriesFor: #newSocketPair!private!services! !
!SpSocket class categoriesFor: #newTCPClientSocket!public! !
!SpSocket class categoriesFor: #newTCPServerSocket!public!services! !
!SpSocket class categoriesFor: #newTCPSocket!public!services! !
!SpSocket class categoriesFor: #on:!public! !
!SpSocket class categoriesFor: #onNativeclientSocket:for:!private! !
!SpSocket class categoriesFor: #port:!public!services! !
!SpSocket class categoriesFor: #port:backlog:!public!services! !
!SpSocket class categoriesFor: #serverSocketClass!constants!private!services! !

SpSocketAddress guid: (GUID fromString: '{6837112A-192D-4D57-BE2D-0F3B20A93495}')!
SpSocketAddress comment: ''!
!SpSocketAddress categoriesForClass!Unclassified! !
!SpSocketAddress class methodsFor!

on: subjectAddress
	"Well, in the future there may be more than one kind of socket address,
	but for now there is just OSkIPAddress, so I return one of those on the 
	details embodied in the subjectAddress. 
	No use is made of aSocket as yet, but it will be useful when there
	are more kinds of socket address supported."

	^SpIPAddress host: subjectAddress host port: nil!

on: subjectAddress for: aSocket 
	"^an OSkSocketAddress
	Well, in the future there may be more than one kind of socket address,
	but for now there is just OSkIPAddress, so I return one of those on the 
	details embodied in the subjectAddress. 
	No use is made of aSocket as yet, but it will be useful when there
	are more kinds of socket address supported."

	#eamChanged.
	^SpIPAddress host: subjectAddress ipAddress port: aSocket underlyingSocket port! !
!SpSocketAddress class categoriesFor: #on:!public! !
!SpSocketAddress class categoriesFor: #on:for:!instance creation!private! !

SpStringUtilities guid: (GUID fromString: '{10ECB9DD-969F-427B-8CD4-03AD76BBC68B}')!
SpStringUtilities comment: ''!
!SpStringUtilities categoriesForClass!Unclassified! !
!SpStringUtilities class methodsFor!

bytes: subjectBytes asStringUsingEncodingNames: anEncodingName 
	^subjectBytes asStringEncoding: anEncodingName!

prevIndexOf: anElement from: startIndex to: stopIndex in: aString
	"Answer the previous index of anElement within the receiver between startIndex and stopIndex
	 working backwards through the receiver.  If the receiver does not contain anElement, answer nil"

	startIndex to: stopIndex by: -1 do:
		[:i |
		(aString at: i) = anElement
			ifTrue: [^i]].
	^nil!

string: subjectString asBytesUsingEncodingNamed: anEncodingName 
	^subjectString asByteArrayEncoding: anEncodingName!

stringFromBytes: aByteArray 
	"^a String
	In GemStone ['Hello, World' asByteArray asString] returns the string 'aByteArray' !!
	This is the boring long way of getting a string from a ByteArray - but it does work
	in GemStone."
	"SpStringUtilities stringFromBytes: ('Hello, World' asByteArray)"

	^aByteArray asString
!

tokensBasedOn: separatorString in: aString 
	"Answer an OrderedCollection of the sub-sequences
	 of the receiver that are separated by anObject."

	^aString subStrings: separatorString!

trimBlanksFrom: aString 
	"I return a copy of aString with all leading and trailing blanks removed."

	^aString trimBlanks! !
!SpStringUtilities class categoriesFor: #bytes:asStringUsingEncodingNames:!private!services-encoding! !
!SpStringUtilities class categoriesFor: #prevIndexOf:from:to:in:!private!services! !
!SpStringUtilities class categoriesFor: #string:asBytesUsingEncodingNamed:!private!services-encoding! !
!SpStringUtilities class categoriesFor: #stringFromBytes:!public!services-encoding! !
!SpStringUtilities class categoriesFor: #tokensBasedOn:in:!public!services! !
!SpStringUtilities class categoriesFor: #trimBlanksFrom:!public!services! !

SpTimestamp guid: (GUID fromString: '{12BF8FBC-127D-469B-BCCA-9214B732537E}')!
SpTimestamp comment: ''!
!SpTimestamp categoriesForClass!Unclassified! !
!SpTimestamp methodsFor!

< aSpTimestamp
	"Answer true if receiver is less than aSpTimestamp."

	#eamAdded.
	^self underlyingTimestamp  < aSpTimestamp underlyingTimestamp!

<= anOSkTimeStamp 
	^self underlyingTimestamp <= anOSkTimeStamp underlyingTimestamp!

= anOSkTimeStamp
^self underlyingTimestamp = anOSkTimeStamp underlyingTimestamp!

> anOSkTimeStamp
^self underlyingTimestamp > anOSkTimeStamp underlyingTimestamp!

>= aSpTimestamp
	"Answer whether receiver is greather or equal than aSpTimestamp."

	#eamAdded.
	^(self < aSpTimestamp) == false!

asNowUTC
	"^self
Cheat for now and assumen that Timestamp>>now is UTC."

	underlyingTimestamp := TimeStamp current.
	^self!

asRFC1123String
	"^a String
	c.f  >>asRFC1123StringOn: "

	#eamAdded.
	^self rfc1123String!

asRFC1123StringOn: targetStream
	"Assumes server's clock is GMT.  Should convert server's clock to GMT if it is not.  Besides that, this whole method is really ugly."

	^self underlyingTimestamp printRFC1123StringOn: targetStream!

asSeconds
	"^an Integer
	I return the timestamp as a number of seconds."
	^self underlyingTimestamp asSeconds!

asSpTimestamp
	^self!

asTime
	^self underlyingTimestamp time!

fromDate: aDate andTime: aTime
	"^self
	Initialize myself on the basis of aDate and aTime."
	underlyingTimestamp := TimeStamp date: aDate time: aTime.
	^self!

fromSeconds: anInteger 
	"^a SpTimestamp
	I return an instance of myself that represents anInteger number of seconds 
	since January 1, 1901 0:00:00.000.  BTW, negative values of anInteger are fine."
	
	underlyingTimestamp := TimeStamp fromSeconds: anInteger.
	^self!

hash
	^self underlyingTimestamp hash!

printString
	^self underlyingTimestamp printString!

rfc1123String
	"Answer a string representing receiver in RFC1123 date and time format.
	E.g. Sun, 06 Nov 1994 08:49:37 GMT . "
	
	#eamAdded.
	^self underlyingTimestamp rfc1123String!

underlyingTimestamp
	^underlyingTimestamp! !
!SpTimestamp categoriesFor: #<!public! !
!SpTimestamp categoriesFor: #<=!comparing!public! !
!SpTimestamp categoriesFor: #=!comparing!public! !
!SpTimestamp categoriesFor: #>!comparing!public! !
!SpTimestamp categoriesFor: #>=!public! !
!SpTimestamp categoriesFor: #asNowUTC!initialize-release!private! !
!SpTimestamp categoriesFor: #asRFC1123String!private!services! !
!SpTimestamp categoriesFor: #asRFC1123StringOn:!public!services! !
!SpTimestamp categoriesFor: #asSeconds!converting!public! !
!SpTimestamp categoriesFor: #asSpTimestamp!converting!public! !
!SpTimestamp categoriesFor: #asTime!converting!public! !
!SpTimestamp categoriesFor: #fromDate:andTime:!initialize-release!public! !
!SpTimestamp categoriesFor: #fromSeconds:!initialize-release!public! !
!SpTimestamp categoriesFor: #hash!comparing!public! !
!SpTimestamp categoriesFor: #printString!private! !
!SpTimestamp categoriesFor: #rfc1123String!public!services! !
!SpTimestamp categoriesFor: #underlyingTimestamp!private! !

!SpTimestamp class methodsFor!

fromDate: aDate andTime: aTime 
	^self new fromDate: aDate andTime: aTime!

fromRFC1123String: aString 
	"Answer an instance of receiver built from aString in RFC1123 Date and time format.
	E.g.: 'Sun, 06 Nov 1994 08:49:37 GMT'"

	| sourceStream dd mmm yyyy time |
	#eamAdded.
	^SpExceptionContext for: 
			[sourceStream := ReadStream on: aString.
			sourceStream upTo: Character space.
			dd := sourceStream upTo: Character space.
			mmm := sourceStream upTo: Character space.
			yyyy := sourceStream upTo: Character space.
			time := sourceStream upTo: Character space.
			self fromDate: (Date 
						newDay: dd asNumber
						monthIndex: (self monthIndexOfRFC1123Month: mmm)
						year: yyyy asNumber)
				andTime: (Time readFrom: (ReadStream on: time))]
		onAnyExceptionDo: 
			[:exception | 
			SpError raiseSignal: 'Error parsing RFC1123 date: ' , aString]!

fromSeconds: anInteger
	"^a SpTimestamp
	I return an instance of myself that represents anInteger number of seconds since..."
	^self new fromSeconds: anInteger!

fromTimeStamp: aTimeStamp
	"^a SpTimestamp, from a Squeak TimeStamp"
	^self fromDate: aTimeStamp date andTime: aTimeStamp time!

monthIndexOfRFC1123Month: aString 
	"Private - Answer the month index for a month name in RFC1123 format (abbreviated).
	I.e. August is 'Aug'."

	#eamAdded.
	^Locale englishUS monthNames at: aString asLowercase!

now
	"^an OSkTimestamp
	I return a new instance of myself which represents the time now in the
	UTC (GMT ish) time zone."

	^self new asNowUTC! !
!SpTimestamp class categoriesFor: #fromDate:andTime:!instance creation!public! !
!SpTimestamp class categoriesFor: #fromRFC1123String:!instance creation!public! !
!SpTimestamp class categoriesFor: #fromSeconds:!instance creation!public! !
!SpTimestamp class categoriesFor: #fromTimeStamp:!instance creation!public! !
!SpTimestamp class categoriesFor: #monthIndexOfRFC1123Month:!private! !
!SpTimestamp class categoriesFor: #now!instance creation!public! !

SpTranscript guid: (GUID fromString: '{CC6D69ED-AFC7-4644-8845-1E87A1DE086A}')!
SpTranscript comment: ''!
!SpTranscript categoriesForClass!Unclassified! !
!SpTranscript class methodsFor!

cr	
	^ SpEnvironment isHeadless
		ifTrue: [self]
		ifFalse: [Transcript cr]!

nextPut: anObject
	^self show: (String with: anObject)!

nextPutAll: aCollection 
	^self show: aCollection!

show: aString
	^ SpEnvironment isHeadless
		ifTrue: [self]
		ifFalse: [Transcript show: aString]! !
!SpTranscript class categoriesFor: #cr!logging!public! !
!SpTranscript class categoriesFor: #nextPut:!logging!public! !
!SpTranscript class categoriesFor: #nextPutAll:!logging!public! !
!SpTranscript class categoriesFor: #show:!logging!public! !

SpWeakArray guid: (GUID fromString: '{84B49318-2B9A-45FE-AA00-8AA4F7933D77}')!
SpWeakArray comment: ''!
!SpWeakArray categoriesForClass!Unclassified! !
!SpWeakArray class methodsFor!

new: anInteger
	"^a WeakArray
	I don't return an instance of myself, I return a real WeakArray."
	^WeakArray new: anInteger!

withAll: aCollection 
	"^a WeakArray 
	I don't return an instance of myself at all. I return a real Weak array."

	^WeakArray withAll: aCollection! !
!SpWeakArray class categoriesFor: #new:!instance creation!public! !
!SpWeakArray class categoriesFor: #withAll:!instanceCreation!private! !

SpAbstractError guid: (GUID fromString: '{F99302F8-E5DA-434C-8B80-C3C4EA7C047D}')!
SpAbstractError comment: ''!
!SpAbstractError categoriesForClass!Unclassified! !
!SpAbstractError methodsFor!

errorString
	^self messageText!

isResumable
	"Determine whether an exception is resumable."

	^self class mayResume!

parameter
	^parameter !

parameter: anObject 
	parameter := anObject!

raiseSignal
	"Raise an an exception."
	^ self signal!

raiseSignal: aString 
	"Raise an an exception."
	^ self signal: aString! !
!SpAbstractError categoriesFor: #errorString!accessing!public! !
!SpAbstractError categoriesFor: #isResumable!priv handling!public! !
!SpAbstractError categoriesFor: #parameter!accessing!public! !
!SpAbstractError categoriesFor: #parameter:!accessing!public! !
!SpAbstractError categoriesFor: #raiseSignal!public!signalling! !
!SpAbstractError categoriesFor: #raiseSignal:!public!signalling! !

!SpAbstractError class methodsFor!

mayResume

	^false!

raiseSignal
	"Raise an an exception."
	^ self signal!

raiseSignal: aString 
	"Raise an an exception."
	^ self signal: aString! !
!SpAbstractError class categoriesFor: #mayResume!public!testing! !
!SpAbstractError class categoriesFor: #raiseSignal!public!signalling! !
!SpAbstractError class categoriesFor: #raiseSignal:!public!signalling! !

SpError guid: (GUID fromString: '{A4B98E7A-AB5B-4D8F-A9C0-B24534C045A8}')!
SpError comment: ''!
!SpError categoriesForClass!Unclassified! !
SpSocketError guid: (GUID fromString: '{198F5F01-6526-4217-935A-ADE88E86B4EB}')!
SpSocketError comment: ''!
!SpSocketError categoriesForClass!Unclassified! !
SpIPAddress guid: (GUID fromString: '{8F8991B8-9DD2-449A-939B-A7776F939192}')!
SpIPAddress comment: ''!
!SpIPAddress categoriesForClass!Unclassified! !
!SpIPAddress methodsFor!

= aSpIPAddress
	"Answer whether receiver is equal to aSpIPAddress."

	#eamAdded.
	^self hostAddress ipAddress = aSpIPAddress hostAddress ipAddress and: [self portNumber = aSpIPAddress portNumber]!

asNativeSocketAddress
	"^an IPSocketAddress
	I return a VisualWorks IPSocketAddress which represents the same IP address I do."

	^Socket2 port: self portNumber address: (InternetAddress ipAddress: self hostAddress) !

hash
	"Answer hash number of receiver."

	#eamAdded.
	^self hostAddress ipAddress hash bitXor: self portNumber !

host: aHostAddress port: aPortNumber 
	hostAddress := aHostAddress.
	portNumber := aPortNumber!

hostAddress
	^ hostAddress!

hostAddressString
	| targetStream |
	targetStream := String new writeStream.

	targetStream
		nextPutAll: (self hostAddress at: 1) printString;
		nextPut: $.;
		nextPutAll: (self hostAddress at: 2) printString;
		nextPut: $.;
		nextPutAll: (self hostAddress at: 3) printString;
		nextPut: $.;
		nextPutAll: (self hostAddress at: 4) printString.
	^targetStream contents!

portNumber
	^ portNumber! !
!SpIPAddress categoriesFor: #=!public! !
!SpIPAddress categoriesFor: #asNativeSocketAddress!private!services - private! !
!SpIPAddress categoriesFor: #hash!public! !
!SpIPAddress categoriesFor: #host:port:!initialize-release!public! !
!SpIPAddress categoriesFor: #hostAddress!accessing!public! !
!SpIPAddress categoriesFor: #hostAddressString!printing!public! !
!SpIPAddress categoriesFor: #portNumber!accessing!public! !

!SpIPAddress class methodsFor!

host: aHostAddress port: aPortNumber
	"^a SpSocketAddress
	I create a new instance of myself which represents an IP address/port
	combination (a TCP/IP address, really). Note that aHostAddress must be a
	four element byte array (e.g. #[127 0 0 1]) ."
	
	^ self new host: aHostAddress port: aPortNumber!

hostName: aHostNameString port: aPortNumber 
	"^an OSkSocketAddress
	I translate aHostNameString to an IP address and then create
	a new instance of myself with >>host:port:"

	^self host: (InternetAddress host: aHostNameString) port: aPortNumber! !
!SpIPAddress class categoriesFor: #host:port:!instance creation!public! !
!SpIPAddress class categoriesFor: #hostName:port:!private!services! !

"Binary Globals"!

